const sendResponse = require("../Helper/Helper");
const EmployeeModel = require("../models/EmployeeModel");
const multer = require("multer");
const checkEmailValidity = require("../Helper/checkEmail");
var dotenv = require("dotenv");
const Hash = require("../Helper/hashing");
const Shopify = require("shopify-api-node");
const StoreSchema = require("../models/StoreSchema");
const { default: mongoose } = require("mongoose");
dotenv.config();

const AuthController = {
  getEmployees: async (req, res) => {
    try {
      let { page, limit, employeeAssociation } = req.query;

      if (!page) page = 1;
      if (!limit) limit = 50;

      const skip = (page - 1) * limit;

      if (!employeeAssociation) {
        res
          .send(sendResponse(false, null, "Required employee shop domain"))
          .status(400);
        return;
      }
      const result = await EmployeeModel.find({
        employeeAssociation,
      })
        .skip(skip)
        .limit(limit)
        .populate("employeeOrganization");

      if (!result) {
        res
          .send(sendResponse(false, null, "No Employee Data Found"))
          .status(404);
      } else {
        res.send(sendResponse(true, result, "Employee Data Found")).status(200);
      }
    } catch (error) {
      console.log(error);
      res
        .send(sendResponse(false, null, "Server Internal Error", error))
        .status(400);
    }
  },
  addEmployee: async (req, res, reuse = false) => {
    let {
      email,
      userCapTotal,
      employeeAssociation,
      discountType,
      discountValue,
      discountFormat,
      CollectionId,
      employeeOrganization,
    } = req.body;
    try {
      let errArr = [];

      //validation part
      if (!email) {
        errArr.push("Required employee email");
      }
      if (discountFormat && !CollectionId) {
        errArr.push("Required Collection Id");
      }

      if (!userCapTotal) {
        errArr.push("Required employee Total Cap");
      }
      if (!discountValue) {
        errArr.push("Required employee discount Value");
      }
      if (!discountType) {
        errArr.push("Required discount type");
      }

      if (!employeeAssociation) {
        errArr.push("Required employee association");
      }

      const checkEmail = checkEmailValidity(email);
      if (!checkEmail.status) {
        errArr.push(`${checkEmail.message}`);
      }

      if (errArr.length > 0) {
        if (reuse) {
          return { data: null, message: "Required All Fields", status: false };

          // return "Required All Fields";
        } else {
          res
            .send(sendResponse(false, errArr, null, "Required All Fields"))
            .status(400);
          return;
        }
      } else {
        let userExist = await EmployeeModel.findOne({ email: email });

        if (userExist) {
          if (reuse) {
            return {
              data: null,
              message: "User with this email already exist",
              status: false,
            };

            // return "User with this email already exist";
          } else {
            res
              .send(
                sendResponse(false, null, "User with this email already exist")
              )
              .status(400);
            return;
          }
        } else {
          //add new user to mongodb
          let obj = {
            email,
            userCapTotal,
            employeeAssociation,
            discountType,
            discountValue,
          };
          obj = {
            ...obj,
            userCapRemain: userCapTotal,
            ...(discountFormat && {
              discountInfo: {
                format: discountFormat,
                // Only add collectionIds if the format is "Collection"
                ...(discountFormat === "Collection" && {
                  collectionIds: CollectionId.map((item) => parseInt(item)), // Only include if format is "Collection"
                }),
              },
            }),
            ...(employeeOrganization && {
              employeeOrganization,
            }),
          };

          // console.log("object", obj);
          let Employee = new EmployeeModel(obj);
          await Employee.save();
          // res
          //   .send(sendResponse(true, Employee, "Employee Added Successfully"))
          //   .status(200);
          // return;
          //fetch user id through user email.

          const store = await StoreSchema.findOne({
            shopName: employeeAssociation,
          });

          if (!store) {
            if (reuse) {
              return {
                data: null,
                message: "No credentials found for the shop.",
                status: false,
              };
            } else {
              return res.status(404).send({
                status: false,
                message: "No credentials found for the shop.",
              });
            }
          }

          console.log("first", store);
          const decryptedApiKey = await Hash.decrypt(store.apiKey);
          const decryptedApiSecret = await Hash.decrypt(store.apiSecret);

          if (!employeeAssociation && !decryptedApiKey && !decryptedApiSecret) {
            if (reuse) {
              return {
                data: null,
                message:
                  "No credentials found i.e decrypted api key or api secret",
                status: false,
              };
            } else {
              return res.status(404).send({
                status: false,
                message:
                  "No credentials found i.e decrypted api key or api secret",
              });
            }
          }

          console.log("second", decryptedApiKey, decryptedApiSecret);

          const shopify = new Shopify({
            shopName: employeeAssociation,
            apiKey: decryptedApiKey.trim(),
            password: decryptedApiSecret.trim(),
            apiVersion: "2025-01",
          });

          const user = await shopify.customer.search({
            query: `email:${email}`,
          });
          let payload = Employee.toObject();
          // console.log("checkup", query);
          if (user[0]) {
            payload.customerId = user[0].id;
          } else {
            const query = await shopify.graphql(
              `mutation customerCreate($input: CustomerInput!) {
                customerCreate(input: $input) {
                  userErrors {
                    field
                    message
                  }
                  customer {
                    id
                    email
                    phone
                    taxExempt
                    firstName
                    lastName
                    amountSpent {
                      amount
                      currencyCode
                    }
                  }
                }
              }`,
              {
                input: {
                  email: email,
                  // firstName: email.split("@")[0],
                },
              }
            );
            if (query.customerCreate.userErrors.length > 0) {
              if (reuse) {
                return {
                  status: false,
                  message: JSON.stringify(
                    query.customerCreate.userErrors[0].field.message
                  ),
                };
              } else {
                return res.status(404).send({
                  status: false,
                  message: JSON.stringify(query.customerCreate.userErrors[0]),
                });
              }
            }

            payload.customerId =
              query.customerCreate.customer.id.split("Customer/")[1];
          }

          if (reuse) {
            return {
              data: payload,
              message: "Employee Added Successfully",
              status: true,
            };
          } else {
            res
              .send(sendResponse(true, query, "Employee Added Successfully"))
              .status(200);
          }
        }
      }
    } catch (error) {
      console.log("error reason", error.stack);
      let errorMessage = "An unexpected error occurred";
      if (error.response && error.response.data && error.response.data.errors) {
        // Extract and format dynamic errors
        errorMessage = Object.entries(error.response.data.errors)
          .map(([key, messages]) => `${key}: ${messages}`)
          .join("\n");
      }
      res
        .send(sendResponse(false, null, errorMessage, "Internal Server Error"))
        .status(400);
    }
  },
  editEmployee: async (req, res) => {
    try {
      let {
        email,
        userCapTotal,
        employeeAssociation,
        discountType,
        discountValue,
        discountFormat,
        CollectionId,
      } = req.body;
      let id = req.params.id;
      if (!id) {
        return res.send(sendResponse(false, null, "id not found")).status(404);
      }
      let result = await EmployeeModel.findById(id);
      if (!result) {
        res
          .send(sendResponse(false, null, "Employee with specified id Found"))
          .status(404);
      } else {
        let obj = {
          email,
          userCapTotal,
          employeeAssociation,
          discountType,
          discountValue,
        };
        obj = {
          ...obj,
          userCapRemain: userCapTotal,
          ...(discountFormat && {
            discountInfo: {
              format: discountFormat,
              // Only add collectionIds if the format is "Collection"
              ...(discountFormat === "Collection" && {
                collectionIds: CollectionId.map((item) => parseInt(item)), // Only include if format is "Collection"
              }),
            },
          }),
        };

        let updateResult = await EmployeeModel.findByIdAndUpdate(id, obj, {
          new: true,
        });
        if (!updateResult) {
          res
            .send(sendResponse(false, null, "Unable to update results"))
            .status(404);
        } else {
          res
            .send(
              sendResponse(
                true,
                updateResult,
                "Employee data updated SucessFully"
              )
            )
            .status(200);
        }
      }
    } catch (e) {
      res
        .send(sendResponse(false, null, e, "Internal Server Error"))
        .status(400);
    }
  },
  deleteEmployee: async (req, res) => {
    try {
      let id = req.params.id;
      if (!id) {
        return res.send(sendResponse(false, null, "id not found")).status(404);
      }
      let result = await EmployeeModel.findById(id);
      // // console.log("resultant", result);
      // if (!id) {
      //   return res.send(sendResponse(false, null, "employee with specified id not found")).status(404);
      // }
      if (!result) {
        res
          .send(sendResponse(false, null, "Employee with specified id Found"))
          .status(404);
      } else {
        let deleteById = await EmployeeModel.findByIdAndDelete(id);
        if (!deleteById) {
          res.send(sendResponse(false, null, "Error")).status(404);
        } else {
          res
            .send(
              sendResponse(
                true,
                deleteById,
                "Employee Data Deleted SucessFully"
              )
            )
            .status(200);
        }
      }
    } catch (e) {
      res
        .send(sendResponse(false, null, e, "Internal Server Error"))
        .status(400);
    }
  },
  getSingleEmployees: async (req, res) => {
    let id = req.params.id;
    if (!id) {
      return res.send(sendResponse(false, null, "id not found")).status(404);
    }
    await EmployeeModel.findById(id)
      .then((result) => {
        res.send(sendResponse(true, result)).status(200);
      })
      .catch((err) => {
        console.log(err);
        res.send(sendResponse(false, err, "Employee not found")).status(404);
      });
  },
  getSingleOrganizationEmployees: async (req, res) => {
    let id = req.params.orgId;
    if (!id) {
      return res
        .send(sendResponse(false, null, "organization id not found"))
        .status(404);
    }
    // let result = await EmployeeModel.find({ employeeOrganization: id });
    let result = await EmployeeModel.aggregate([
      {
        $match: { employeeOrganization: new mongoose.Types.ObjectId(id) },
      },
      {
        $project: {
          email: 1,
        },
      },
    ]);

    if (result) {
      res.send(sendResponse(true, result)).status(200);
    } else {
      res.send(sendResponse(false, err, "Organization not found")).status(404);
    }
  },
};

module.exports = AuthController;
